function [oprice,Delta, Vega,Gamma, xi,eta] = LN_f(Ft_vec, strike, rf,TTM, sigma_vec,isgrad,ishess)
%==========================================================================================
%This function computes the black call option price and related derivatives based on a
%Lognormal distribution. 
%
%INPUT:
%   Ft_vec: K-by-1 vector of mean/futures values of each Lognormal density
%   strike: N-by-1 strike prices of the options
%   rf: risk-free rate
%   TTM: time to maturity of the options (in years)
%   sigma_vec: K-by-1 vector of sigma values of each Lognormal density
%   isgrad: logical, whether gradients (first derivatives Greeks) 
%           are computed. Default = false
%   ishess: logical, whether hessians (second derivatives Greeks) 
%           are computed. Default = false
%
%OUTPUT:
%   oprice: N-by-K call option prices implied by each Lognormal density for
%           each strike
%   Delta,Vega,Gamma,xi,eta: N-by-K matrices of option Greeks
%==========================================================================================
% This ver: 2023/05/24
% Authors: Yifan Li (yifan.li@manchester.ac.uk)
%          Ingmar Nolte (i.nolte@lancaster.ac.uk)
%          Manh Pham (m.c.pham@lancaster.ac.uk)
% Reference: Li, Y., Nolte, I., and Pham, M. C. (2023). Parametric Risk-Neutral 
%          Density Estimation via Finite Lognormal-Weibull Mixtures
%========================================================================================== 

if nargin <6
    isgrad = false; ishess=false;
elseif nargin <7 
    ishess=false;
end

%Black pricing formula
d1 = (log(Ft_vec./strike) + (0.5*sigma_vec.^2).*TTM)./(sigma_vec.*sqrt(TTM));
d2 = d1 - sigma_vec.*sqrt(TTM);
Nd1 = normcdf(d1);
Nd2 = normcdf(d2);
discount = exp(-rf.*TTM);
oprice = discount.*(Nd1.*Ft_vec - Nd2.*strike);
if isgrad
    nd1 = normpdf(d1);
    Delta = discount.*Nd1;
    Vega = discount.*Ft_vec.*nd1.*sqrt(TTM);   
    if ishess
        Gamma = discount.*nd1./Ft_vec./sqrt(TTM)./sigma_vec;
        xi=Vega.*d1.*d2./sigma_vec;
        eta=-discount.*nd1.*d2./sigma_vec;
    else
        Gamma = []; xi =[]; eta = [];
    end
    else
    Delta = []; Vega = [];
end
end

